#!/usr/bin/env python3
# -*- coding: utf-8 -*-

"""
This file contains the core code for the Flappy Bird game, implemented using the pygame library
(https://www.pygame.org/).
This code was inspired by https://github.com/clear-code-projects/FlappyBird_Python
"""

from random import randint

import pygame


class FlappyGame:
  """The Flappy Bird game class"""

  def __init__(self, window_size, interactive, fps, hard_version=False):
    # The game constants
    if hard_version:
      self.GRAVITY = 600
      self.SPEED_ON_FLAP = 300
      self.HORIZONTAL_SPEED = 240
      self.VERTICAL_SPACE = 50
      self.HOURS_EVERY_SECOND = 0.25
      self.PIPE_LOWEST_POS = 350
      self.PIPE_HIGHEST_POS = 75
    else:
      self.GRAVITY = 720
      self.SPEED_ON_FLAP = 300
      self.HORIZONTAL_SPEED = 180
      self.VERTICAL_SPACE = 125
      self.HOURS_EVERY_SECOND = 0.25
      self.PIPE_LOWEST_POS = 425
      self.PIPE_HIGHEST_POS = 150

    # The size of the game window
    self.window_size = window_size

    # Is the game interactive, i.e., can it be played with the spacebar (or only through external
    # code)?
    self.interactive = interactive

    # The framerate of the game
    self.fps = fps

    # Should we play the easy or the hard version?
    self.hard_version = hard_version

    # The delta time, used for framerate-independent physics
    self.dt = 1/fps

    # Is the game still running?
    self.exit = False

    # The high score
    self.high_score = 0

    # We initialize pygame
    pygame.init()
    self.screen = pygame.display.set_mode(window_size)
    self.clock = pygame.time.Clock()

    # We load the textures
    self.bird_texture = pygame.image.load("assets/bird.png").convert()
    self.bg_day_texture = pygame.image.load("assets/background_day_wide.png").convert()
    self.bg_night_texture = pygame.image.load("assets/background_night_wide.png").convert_alpha()
    self.floor_texture = pygame.image.load("assets/floor_wide.png").convert()
    self.pipe_texture = pygame.image.load("assets/pipe_red.png").convert()
    self.flipped_pipe_texture = pygame.transform.flip(self.pipe_texture, False, True)

    # We make the bg_night texture transparent
    self.bg_night_texture.set_alpha(0)

    # We load the font
    self.font = pygame.font.Font("assets/flappy_font.ttf", 20)

    # And we reset the game to its initial state
    self.reset_game()


  def reset_game(self):
    # We set the game to not running
    self.running = False

    # We reset the frame count (the number of times a frame was posted)
    self.frame_count = 0

    # We reset the score
    self.score = 0

    # We set the position of the floor to 0
    self.floor_x_pos = 0

    # We set the position of the bird on the screen to the middle of the window
    self.bird_x_pos = self.window_size[0]/3
    self.bird_y_pos = self.window_size[1]/2

    # We set the bird vertical speed to 0
    self.bird_y_speed = 0.0

    # We clear the list of pipes
    self.pipes_x_pos = []
    self.pipes_y_pos = []

    # We set the current time to 8 am
    self.current_time = 8.0

    # We draw the background
    self.draw_background()

    # We draw the pipes
    self.draw_pipes()

    # We draw the floor
    self.draw_floor()

    # We draw the bird
    self.draw_bird()

    # And we finish by drawing the score
    self.draw_score()


  def update(self):
    # For each event captured by pygame
    for event in pygame.event.get():
      # If the user decides to quit...
      if event.type == pygame.QUIT:
        # We set self.exit and we do not process any other event
        self.exit = True
        return -3

      # If the user pressed the space key (only in interactive mode)...
      if self.interactive and event.type == pygame.KEYDOWN and event.key == pygame.K_SPACE:
        # We make the bird flap once (which also starts the game if it is not yet running)
        self.flap()
      
    # We set by default the score to return to -2 ("game not running")
    score_to_return = -2

    # Then, and only if the game is running...
    if self.running:
      # We set the score to return to -1 ("game running")
      score_to_return = -1

      # We increment the score is needed (+1 every second)
      if (self.frame_count % self.fps) == self.fps - 1:
        self.score += 1

      # We check if the bird is colliding with another object
      is_colliding = self.check_collisions()
      if is_colliding:
        # If it is the case, we update the high score if needed
        if self.score > self.high_score:
          self.high_score = self.score
        
        # We also set the score to return to the score before colliding (>= 0)
        score_to_return = self.score
        
        # And we reset the game
        self.reset_game()

      # We update the background and draw it
      self.update_background()
      self.draw_background()

      # We update the pipes and draw them
      self.update_pipes()
      self.draw_pipes()

      # We update the floor and draw it
      self.update_floor()
      self.draw_floor()

      # We update the bird and draw it
      self.update_bird_position()
      self.draw_bird()

      # We draw the score
      self.draw_score()

    # Once all that is done, we display the new frame, and increase the frame count
    pygame.display.flip()
    self.frame_count += 1

    # And we sleep for the correct amount of time (only if in interactive mode)
    if self.interactive:
      self.dt = self.clock.tick(self.fps) / 1000
    
    # Once done, we return the score (if the bird has collided with something, the score before
    # colliding, otherwise -1 if the game has not been initialized or if it still running)
    return score_to_return


  def check_collisions(self):
    # We check if the bird is colliding with the floor
    if self.bird_rect.bottom >= self.floor_rect.top:
      return True

    # We check if the bird is colliding with the ceiling
    if self.bird_rect.top <= 0:
      return True

    # We check if the bird is colliding with any of the pipes
    for pipe_rect in self.pipes_rect:
      if self.bird_rect.colliderect(pipe_rect):
        return True

    # If we arrive here, the bird is colliding with nothing
    return False


  def update_background(self):
    # We update the current time (between 0.00 and 23.99)
    self.current_time += self.HOURS_EVERY_SECOND * self.dt
    self.current_time %= 24

    # If the current time is over 16.00 or under 8.00, we must set the correct alpha value for the
    # bg_night texture (linear mix until 20.00 and from 4.00); otherwise we set alpha to 0
    if self.current_time > 16:
      self.bg_night_texture.set_alpha(min((self.current_time-16)/4, 1.0)*255)
    elif self.current_time < 8:
      self.bg_night_texture.set_alpha(min((8-self.current_time)/4, 1.0)*255)
    else:
      self.bg_night_texture.set_alpha(0)


  def draw_background(self):
    # We just draw the background textures, on top of each other
    self.screen.blit(self.bg_day_texture, (0, 0))
    self.screen.blit(self.bg_night_texture, (0, 0))


  def update_pipes(self):
    # We update the x position of each pipe
    for i, _ in enumerate(self.pipes_x_pos):
      self.pipes_x_pos[i] -= self.HORIZONTAL_SPEED * self.dt

    # If the first pipe is out of the screen on the left, we delete it
    if self.pipes_x_pos and self.pipes_x_pos[0] <= -100:
      self.pipes_x_pos.pop(0)
      self.pipes_y_pos.pop(0)

    # If there is not yet a single pipe, or if there is less than 4 pipes and the 3rd one is far
    # enough...
    if not self.pipes_x_pos or (len(self.pipes_x_pos) < 4 and self.pipes_x_pos[-1] <= 500):
      # We add a new pipe, just out of the screen, and at a random height
      self.pipes_x_pos.append(self.screen.get_width())
      self.pipes_y_pos.append(randint(self.PIPE_HIGHEST_POS, self.PIPE_LOWEST_POS))


  def draw_pipes(self):
    # We create a list which will contain the hitbox of each pipe
    self.pipes_rect = []

    # Then, for each pipe...
    for pipe_x_pos, pipe_y_pos in zip(self.pipes_x_pos, self.pipes_y_pos):
      # We draw the bottom pipe
      self.pipes_rect.append(self.screen.blit(self.pipe_texture, (pipe_x_pos, pipe_y_pos)))

      # We draw the top pipe
      self.pipes_rect.append(self.screen.blit(self.flipped_pipe_texture,
                                              (pipe_x_pos, pipe_y_pos-self.VERTICAL_SPACE-self.pipe_texture.get_height())))


  def update_floor(self):
    # We update the position of the floor, to make it scroll to the left
    self.floor_x_pos -= self.HORIZONTAL_SPEED * self.dt
    self.floor_x_pos %= self.floor_texture.get_width()


  def draw_floor(self):
    # We draw the floor, in two parts to cover the whole screen
    self.floor_rect = self.screen.blit(self.floor_texture, (self.floor_x_pos,
                                                            self.window_size[1]-self.floor_texture.get_height()/2))
    self.screen.blit(self.floor_texture, (self.floor_x_pos-self.floor_texture.get_width(),
                                          self.window_size[1]-self.floor_texture.get_height()/2))


  def update_bird_position(self):
    # We update the speed of the bird, based on the gravity and the elapsed time
    self.bird_y_speed += self.GRAVITY * self.dt

    # And then, we update the position of the bird, based on the speed and the elapsed time
    self.bird_y_pos += self.bird_y_speed * self.dt


  def draw_bird(self):
    # We simply draw the bird at its correct position
    self.bird_rect = self.screen.blit(self.bird_texture, (self.bird_x_pos-self.bird_texture.get_width()/2,
                                                          self.bird_y_pos-self.bird_texture.get_height()/2))
  

  def draw_score(self):
    # We display the high score
    high_score_surface = self.font.render(f"High score: {self.high_score}", False, "black")
    self.screen.blit(high_score_surface, (10, 10))

    # If the game is running, we also display the current score
    if self.running:
      score_surface = self.font.render(f"Score: {self.score}", False, "black")
      self.screen.blit(score_surface, (10, 40))


  def draw_circle(self, pos_x, pos_y, size):
    # We draw a red circle at the desired position and with the desired size (this can be used for
    # debug, for instance)
    pygame.draw.circle(self.screen, "red", (pos_x, pos_y), size)


  def get_bird_y_speed(self):
    # We return the speed of the bird in px/frame
    return self.bird_y_speed * self.dt


  def flap(self):
    # If the game is not running, we start it
    if not self.running:
      self.running = True

    # When flapping, the speed of the bird is set to a large negative value, to make it go up
    self.bird_y_speed = -self.SPEED_ON_FLAP
