import math
import ipyleaflet
from ipyleaflet import Map, Marker, basemaps, Icon, LayerGroup
import ipywidgets
import yaml
import pyproj
import numpy as np
import random

default_center = [44.91261,6.28933]
default_zoom = 12
default_dimensions = ipywidgets.Layout(width='100%', height="800px")

def ouvrir_carte():
    return Map(
        basemap=basemaps.OpenTopoMap, 
        center = default_center, 
        zoom = default_zoom, 
        layout = default_dimensions
    )

latlon_to_mercator = pyproj.Transformer.from_crs("EPSG:4326", "EPSG:3857")

def mercator_from_latlon(lat, lon):
    return latlon_to_mercator.transform(lat, lon)

def latlon_from_mercator(mx, my):
    return latlon_to_mercator.transform(mx, my, direction = 'INVERSE')
    
def distance(lat1, lon1, lat2, lon2):
	rad = math.pi / 180
	rlat1 = lat1 * rad
	rlat2 = lat2 * rad
	sinDLat = math.sin((lat2 - lat1) * rad / 2)
	sinDLon = math.sin((lon2 - lon1) * rad / 2)
	a = sinDLat * sinDLat + math.cos(rlat1) * math.cos(rlat2) * sinDLon * sinDLon
	c = 2 * math.atan2(math.sqrt(a), math.sqrt(1 - a))
	return 6371000 * c

def mercator_distorsion(lat, lon):
    x, y = mercator_from_latlon(lat, lon)
    latx, lonx = latlon_from_mercator(x+1, y)
    laty, lony = latlon_from_mercator(x, y+1)
    rx = distance(lat, lon, latx, lonx)
    ry = distance(lat, lon, laty, lony)
    return rx, ry

def coordonnees(lat, lon):
    dx, dy = mercator_distorsion(lat, lon)
    mx, my = mercator_from_latlon(lat, lon)
    cx, cy = mercator_from_latlon(*default_center)
    return (mx - cx)*dx, (my - cy)*dy

def latlon(x, y):
    lat, lon = default_center
    cx, cy = mercator_from_latlon(*default_center)
    score = None
    while True:
        dx, dy = mercator_distorsion(lat, lon)
        mx = x / dx + cx
        my = y / dy + cy
        nlat, nlon = latlon_from_mercator(mx, my)
        nx, ny = coordonnees(nlat, nlon)
        nscore = (nx-x)*(nx-x) + (ny-y)*(ny-y)
        if score is not None and nscore >= score:
            return lat, lon
        score, lat, lon = nscore, nlat, nlon

def charger_sommets(fichier):
    with open(fichier, 'r') as file:
        return yaml.safe_load(file)

def sauver_sommets(sommets, fichier):
    with open(fichier, 'w') as file:
        file.write(yaml.dump(sommets))

def sauver_sommet(fichier, curseur, name):
    with open(fichier, 'a') as file:
        file.write(yaml.dump({name : coordonnees(curseur.location)}))

def marquer_sommets(carte, sommets, noms = None):
    epingles = LayerGroup()
    for name,location in sommets.items():
        if noms is None or name in noms:
            epingles.add(Marker(
                name = name, 
                location = latlon(*location), 
                title = name,
                draggable = False
                )
            )
    carte.add(epingles)

def degres(rad):
    return 180 * rad / math.pi

def radians(deg):
    return math.pi * deg / 180

def angle(x1, y1, x2, y2, x3, y3):
    xy1 = np.array([x1 - x2, y1 - y2])
    xy3 = np.array([x3 - x2, y3 - y2])
    sc1 = xy1 / np.linalg.norm(xy1)
    sc3 = xy3 / np.linalg.norm(xy3)
    cos = sc1.dot(sc3)
    sin = np.linalg.det([sc1, sc3])
    angle = degres(np.math.atan2(sin, cos))
    if angle < 0:
        angle = angle + 360
    return angle

def generer_table(sommets, nb = 5, rotation = True, selection = None):
    if selection is None:
        selection = random.sample(list(sommets.keys()), nb)
    weights = [random.random() for _ in range(nb)]
    total = sum(weights)
    weights = np.array([w / total for w in weights])
    coords = np.array([sommets[name] for name in selection])
    center = np.sum(coords * weights[:, None], axis = 0)
    displacement = np.random.normal(size = 2)
    center += random.random() * 10000 * displacement / np.linalg.norm(displacement)
    ex = center + np.array([1,0])
    r = 0
    if rotation:
        r = random.random()*360
    angles = [(angle(*ex, *center, *pt) + r) % 360 for pt in coords]

    return dict(sommets = selection, angles = angles)
    

def afficher_table(carte, sommets, table, cx, cy, angle):
    radius = 800

    center_ll = latlon(cx, cy)

    circle = ipyleaflet.Circle(location=center_ll, radius = radius)
    circle.color = "green"
    circle.fill_color = "green"
    carte.add(circle)

    rays = ipyleaflet.Polyline(locations = [[center_ll, latlon(*sommets[s])] for s in table["sommets"]])
    carte.add(rays)

    offsets = [np.array([math.cos(radians(a + angle)), math.sin(radians(a + angle))]) for a in table["angles"]]
    pts = [np.array([cx, cy]) + radius * o for o in offsets]
    marks = [ipyleaflet.CircleMarker(location = latlon(*pt), radius = 4) for pt in pts]
    for m, name in zip(marks, table["sommets"]):
        m.popup = ipywidgets.HTML()
        m.popup.value = name
        m.fill_color = "black"
        m.color = "black"
        m.fill_opacity = 1
        carte.add(m)